<?php
session_start();
if (!isset($_SESSION['username']) || $_SESSION['role'] !== 'admin') {
    header("Location: index.php");
    exit;
}

include 'config.php';

// Filter data
$filter_tanggal = isset($_GET['tanggal']) ? $_GET['tanggal'] : '';
$filter_kelas = isset($_GET['kelas']) ? $_GET['kelas'] : '';
$filter_status = isset($_GET['status']) ? $_GET['status'] : '';

// Query data bimbingan konseling
$query = "SELECT bk.*, s.nama as siswa_nama, s.kelas, s.nis, 
                 wk.nama_wali as wali_nama
          FROM bimbingan_konseling bk 
          JOIN siswa s ON bk.siswa_id = s.id 
          LEFT JOIN wali_kelas wk ON s.kelas = wk.kelas 
          WHERE 1=1";

$params = [];
if (!empty($filter_tanggal)) {
    $query .= " AND DATE(bk.tanggal) = ?";
    $params[] = $filter_tanggal;
}
if (!empty($filter_kelas)) {
    $query .= " AND s.kelas = ?";
    $params[] = $filter_kelas;
}
if (!empty($filter_status)) {
    $query .= " AND bk.status = ?";
    $params[] = $filter_status;
}

$query .= " ORDER BY bk.tanggal DESC, bk.created_at DESC";

// Eksekusi query
$bimbingan = [];
if (!empty($params)) {
    $stmt = mysqli_prepare($conn, $query);
    if ($stmt) {
        $types = str_repeat('s', count($params));
        mysqli_stmt_bind_param($stmt, $types, ...$params);
        mysqli_stmt_execute($stmt);
        $result = mysqli_stmt_get_result($stmt);
        while ($row = mysqli_fetch_assoc($result)) {
            $bimbingan[] = $row;
        }
    }
} else {
    $result = mysqli_query($conn, $query);
    while ($row = mysqli_fetch_assoc($result)) {
        $bimbingan[] = $row;
    }
}

// Hitung statistik
$total_bimbingan = count($bimbingan);
$status_count = array_count_values(array_column($bimbingan, 'status'));

// Query untuk filter kelas
$kelas_query = mysqli_query($conn, "SELECT DISTINCT kelas FROM siswa ORDER BY kelas");
?>

<!DOCTYPE html>
<html lang="id">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Riwayat Bimbingan Konseling - Admin</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.2/dist/css/bootstrap.min.css" rel="stylesheet">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.5.2/css/all.min.css">
    <style>
        .table-responsive {
            max-height: 600px;
            overflow-y: auto;
        }
        .badge-pending {
            background: #ffc107;
            color: #000;
        }
        .badge-approved {
            background: #17a2b8;
            color: #fff;
        }
        .badge-completed {
            background: #28a745;
            color: #fff;
        }
        .badge-rejected {
            background: #dc3545;
            color: #fff;
        }
        .stat-card {
            background: linear-gradient(135deg, #4facfe 0%, #00f2fe 100%);
            color: white;
            border-radius: 10px;
            padding: 15px;
            margin-bottom: 15px;
            text-align: center;
        }
    </style>
</head>
<body class="container py-4">
    <h2 class="text-center mb-4"><i class="fas fa-hands-helping"></i> Riwayat Bimbingan Konseling</h2>

    <a href="dashboard.php" class="btn btn-secondary mb-3">← Kembali ke Dashboard</a>

    <!-- Statistik -->
    <div class="row mb-4">
        <div class="col-md-2">
            <div class="stat-card">
                <h4><?php echo $total_bimbingan; ?></h4>
                <small>Total</small>
            </div>
        </div>
        <div class="col-md-2">
            <div class="stat-card">
                <h4><?php echo $status_count['pending'] ?? 0; ?></h4>
                <small>Pending</small>
            </div>
        </div>
        <div class="col-md-2">
            <div class="stat-card">
                <h4><?php echo $status_count['approved'] ?? 0; ?></h4>
                <small>Approved</small>
            </div>
        </div>
        <div class="col-md-2">
            <div class="stat-card">
                <h4><?php echo $status_count['completed'] ?? 0; ?></h4>
                <small>Completed</small>
            </div>
        </div>
        <div class="col-md-2">
            <div class="stat-card">
                <h4><?php echo $status_count['rejected'] ?? 0; ?></h4>
                <small>Rejected</small>
            </div>
        </div>
    </div>

    <!-- Filter Form -->
    <div class="card mb-4">
        <div class="card-header bg-light">
            <h5 class="mb-0"><i class="fas fa-filter"></i> Filter Data</h5>
        </div>
        <div class="card-body">
            <form method="GET" class="row g-3">
                <div class="col-md-3">
                    <label for="tanggal" class="form-label">Tanggal</label>
                    <input type="date" class="form-control" id="tanggal" name="tanggal" 
                           value="<?php echo $filter_tanggal; ?>">
                </div>
                <div class="col-md-3">
                    <label for="kelas" class="form-label">Kelas</label>
                    <select class="form-control" id="kelas" name="kelas">
                        <option value="">Semua Kelas</option>
                        <?php while($kelas = mysqli_fetch_assoc($kelas_query)): ?>
                            <option value="<?php echo $kelas['kelas']; ?>" 
                                    <?php echo $filter_kelas == $kelas['kelas'] ? 'selected' : ''; ?>>
                                <?php echo $kelas['kelas']; ?>
                            </option>
                        <?php endwhile; ?>
                    </select>
                </div>
                <div class="col-md-3">
                    <label for="status" class="form-label">Status</label>
                    <select class="form-control" id="status" name="status">
                        <option value="">Semua Status</option>
                        <option value="pending" <?php echo $filter_status == 'pending' ? 'selected' : ''; ?>>Pending</option>
                        <option value="approved" <?php echo $filter_status == 'approved' ? 'selected' : ''; ?>>Approved</option>
                        <option value="completed" <?php echo $filter_status == 'completed' ? 'selected' : ''; ?>>Completed</option>
                        <option value="rejected" <?php echo $filter_status == 'rejected' ? 'selected' : ''; ?>>Rejected</option>
                    </select>
                </div>
                <div class="col-md-3 d-flex align-items-end">
                    <button type="submit" class="btn btn-primary w-100">
                        <i class="fas fa-search"></i> Filter
                    </button>
                </div>
            </form>
        </div>
    </div>

    <!-- Tabel Riwayat BK -->
    <div class="card">
        <div class="card-header bg-info text-white">
            <strong><i class="fas fa-history"></i> Data Bimbingan Konseling</strong>
            <span class="badge bg-light text-dark"><?php echo $total_bimbingan; ?> Sesi</span>
        </div>
        <div class="card-body p-0">
            <?php if (empty($bimbingan)): ?>
                <div class="text-center text-muted py-4">
                    <i class="fas fa-comments fa-2x mb-2"></i><br>
                    <strong>Belum ada data bimbingan konseling</strong>
                </div>
            <?php else: ?>
                <div class="table-responsive">
                    <table class="table table-bordered table-hover table-sm mb-0">
                        <thead class="table-light">
                            <tr>
                                <th width="5%">No</th>
                                <th width="10%">Tanggal</th>
                                <th width="15%">Siswa</th>
                                <th width="10%">Kelas</th>
                                <th width="15%">Topik</th>
                                <th width="20%">Keterangan</th>
                                <th width="15%">Wali Kelas</th>
                                <th width="10%">Status</th>
                                <th width="10%">Aksi</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($bimbingan as $index => $data): ?>
                                <tr>
                                    <td class="text-center"><?php echo $index + 1; ?></td>
                                    <td>
                                        <?php echo date('d/m/Y', strtotime($data['tanggal'])); ?>
                                        <br><small class="text-muted"><?php echo date('H:i', strtotime($data['created_at'])); ?></small>
                                    </td>
                                    <td>
                                        <?php echo $data['siswa_nama']; ?>
                                        <br><small class="text-muted">NIS: <?php echo $data['nis']; ?></small>
                                    </td>
                                    <td><?php echo $data['kelas']; ?></td>
                                    <td><?php echo htmlspecialchars($data['topik']); ?></td>
                                    <td><?php echo htmlspecialchars($data['keterangan']); ?></td>
                                    <td><?php echo $data['wali_nama'] ?: '-'; ?></td>
                                    <td class="text-center">
                                        <span class="badge badge-<?php echo $data['status']; ?>">
                                            <?php echo ucfirst($data['status']); ?>
                                        </span>
                                    </td>
                                    <td class="text-center">
                                        <button type="button" class="btn btn-info btn-sm" 
                                                onclick="viewBKDetails(<?php echo $data['id']; ?>)"
                                                title="Lihat Detail">
                                            <i class="fas fa-eye"></i>
                                        </button>
                                        <?php if ($data['status'] == 'pending'): ?>
                                            <button type="button" class="btn btn-warning btn-sm mt-1"
                                                    onclick="updateStatus(<?php echo $data['id']; ?>, 'approved')"
                                                    title="Setujui">
                                                <i class="fas fa-check"></i>
                                            </button>
                                        <?php endif; ?>
                                    </td>
                                </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
            <?php endif; ?>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.2/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        function viewBKDetails(id) {
            // Implementasi view detail BK
            alert('Detail BK ID: ' + id);
        }

        function updateStatus(id, status) {
            if (confirm('Yakin ingin mengubah status?')) {
                // Implementasi update status
                alert('Update status BK ID: ' + id + ' menjadi: ' + status);
            }
        }
    </script>
</body>
</html>